#!/usr/bin/env python3
# Usage: python3 build_tcl.py coreplus_x86.iso coreplus_x86_64.iso

import os
import sys
import shutil
import subprocess
import tempfile
import gzip
from pathlib import Path
from collections import defaultdict
import re

class TinyCoreBuilder:
    def __init__(self, iso_32bit, iso_64bit, output_dir="output", overlay_dir="overlay"):
        self.iso_32bit = Path(iso_32bit)
        self.iso_64bit = Path(iso_64bit)
        self.output_dir = Path(output_dir)
        self.overlay_dir = Path(overlay_dir)
        self.output_dir.mkdir(exist_ok=True)
        
        self.colors = {
            'info': '\033[1;34m',
            'warn': '\033[1;33m', 
            'err': '\033[1;31m',
            'ok': '\033[1;32m',
            'reset': '\033[0m'
        }
    
    def log(self, level, msg):
        color = self.colors.get(level, '')
        print(f"{color}{msg}{self.colors['reset']}")
    
    def extract_iso(self, iso_path, dest_dir):
        iso_path = str(iso_path)
        dest_dir = str(dest_dir)
        
        if shutil.which('7z'):
            subprocess.run(['7z', 'x', '-o' + dest_dir, iso_path], 
                         capture_output=True, check=True)
            return True
        
        if shutil.which('bsdtar'):
            subprocess.run(['bsdtar', '-xf', iso_path, '-C', dest_dir], 
                         capture_output=True, check=True)
            return True
        
        if shutil.which('xorriso'):
            subprocess.run(['xorriso', '-osirrox', 'on', '-indev', iso_path, 
                          '-extract', '/', dest_dir], capture_output=True)
            return True
        
        return False
    
    def find_core_gz(self, iso_dir):
        iso_dir = Path(iso_dir)
        
        candidates = [
            iso_dir / 'boot' / 'core.gz',
            iso_dir / 'boot' / 'corepure64.gz'
        ]
        
        for candidate in candidates:
            if candidate.exists():
                return str(candidate)
        
        for core_file in iso_dir.rglob('core*.gz'):
            return str(core_file)
        
        return None
    
    def extract_core_gz(self, core_gz_path, dest_dir):
        dest_dir = Path(dest_dir)
        dest_dir.mkdir(exist_ok=True)
        
        try:
            with gzip.open(core_gz_path, 'rb') as f_in:
                subprocess.run(['cpio', '-i', '-H', 'newc', '-d'], 
                             input=f_in.read(), cwd=dest_dir, 
                             capture_output=True, check=True)
            return True
        except:
            return False
    
    def get_file_rank(self, filepath, lang, arch):
        filename = Path(filepath).name
        prefix_match = re.match(r'^([erx])_([iax])_(.*)$', filename)
        
        if not prefix_match:
            return 4
        
        file_lang, file_arch, _ = prefix_match.groups()
        
        if file_lang == lang and file_arch == arch:
            return 0
        elif file_lang == 'x' and file_arch == arch:
            return 1
        elif file_lang == lang and file_arch == 'x':
            return 2
        elif file_lang == 'x' and file_arch == 'x':
            return 3
        
        return 5
    
    def get_overlay_key(self, filepath, is_core=False):
        filename = Path(filepath).name
        prefix_match = re.match(r'^[erx]_[iax]_(.*)$', filename)
        
        if prefix_match:
            clean_name = prefix_match.group(1)
        else:
            clean_name = filename
        
        rel_path = Path(filepath).relative_to(self.overlay_dir)
        if is_core:
            rel_path = rel_path.relative_to('core')
        
        return str(rel_path.parent / clean_name)
    
    def find_best_overlays(self, lang, arch):
        best_iso = defaultdict(lambda: (5, None))
        best_core = defaultdict(lambda: (5, None))
        
        if not self.overlay_dir.exists():
            return best_iso, best_core
        
        for file_path in self.overlay_dir.rglob('*'):
            if not file_path.is_file():
                continue
            
            rank = self.get_file_rank(file_path, lang, arch)
            if rank >= 5:
                continue
            
            key = self.get_overlay_key(file_path, is_core=file_path.is_relative_to(self.overlay_dir / 'core'))
            
            if file_path.is_relative_to(self.overlay_dir / 'core'):
                if rank < best_core[key][0]:
                    best_core[key] = (rank, str(file_path))
            else:
                if rank < best_iso[key][0]:
                    best_iso[key] = (rank, str(file_path))
        
        return best_iso, best_core
    
    def apply_overlays(self, iso_dir, core_dir, lang, arch):
        best_iso, best_core = self.find_best_overlays(lang, arch)
        
        self.log('info', f"Applying {len(best_iso)} ISO + {len(best_core)} core overlays")
        
        for key, (rank, src_path) in best_iso.items():
            dest_path = Path(iso_dir) / key
            dest_path.parent.mkdir(parents=True, exist_ok=True)
            shutil.copy2(src_path, dest_path)
            self.log('ok', f"  iso: {Path(src_path).relative_to(self.overlay_dir)} -> {key}")
        
        for key, (rank, src_path) in best_core.items():
            dest_path = Path(core_dir) / key
            dest_path.parent.mkdir(parents=True, exist_ok=True)
            shutil.copy2(src_path, dest_path)
            self.log('ok', f"  core: {Path(src_path).relative_to(self.overlay_dir)} -> {key}")
    
    def repack_core_gz(self, core_dir, core_gz_path):
        import subprocess
        core_dir = Path(core_dir)
        if not any(core_dir.iterdir()):
            self.log('warn', "Core FS directory empty, skipping repack")
            return
        new_path = core_gz_path + '.new'
        try:
            find_proc = subprocess.Popen(['find', '.'], cwd=core_dir, stdout=subprocess.PIPE)
            cpio_proc = subprocess.Popen(['cpio', '-o', '-H', 'newc'], cwd=core_dir, stdin=find_proc.stdout, stdout=subprocess.PIPE)
            
            with gzip.open(new_path, 'wb', compresslevel=9) as f_out:
                shutil.copyfileobj(cpio_proc.stdout, f_out)
            
            cpio_proc.wait()
            find_proc.wait()
            shutil.move(new_path, core_gz_path)
            self.log('ok', f"Repacked {Path(core_gz_path).name} successfully")
        except Exception as e:
            self.log('err', f"Failed to repack core.gz: {e}")

    
    def create_iso(self, iso_dir, output_path, label, isolinux_bin=None, boot_cat=None):
        iso_dir = Path(iso_dir)
        output_path = Path(output_path)
        
        if shutil.which('xorriso'):
            cmd = [
                'xorriso', '-as', 'mkisofs',
                '-l', '-J', '-r', '-V', label
            ]
            
            if isolinux_bin and (iso_dir / isolinux_bin).exists():
                cmd.extend([
                    '-no-emul-boot', '-boot-load-size', '4', '-boot-info-table',
                    '-b', isolinux_bin, '-c', boot_cat
                ])
            
            cmd.extend(['-o', str(output_path), str(iso_dir)])
            subprocess.run(cmd, capture_output=True, check=True)
            
        elif shutil.which('genisoimage'):
            cmd = [
                'genisoimage', '-l', '-J', '-r', '-V', label
            ]
            
            if isolinux_bin and (iso_dir / isolinux_bin).exists():
                cmd.extend([
                    '-no-emul-boot', '-boot-load-size', '4', '-boot-info-table',
                    '-b', isolinux_bin, '-c', boot_cat
                ])
            
            cmd.extend(['-o', str(output_path), str(iso_dir)])
            subprocess.run(cmd, capture_output=True, check=True)
        else:
            raise RuntimeError("Need xorriso or genisoimage to build ISO")
    
    def find_bootloader(self, iso_dir):
        iso_dir = Path(iso_dir)
        
        candidates = [
            ('boot/isolinux/isolinux.bin', 'boot/isolinux/boot.cat'),
            ('isolinux/isolinux.bin', 'isolinux/boot.cat')
        ]
        
        for isolinux, bootcat in candidates:
            if (iso_dir / isolinux).exists():
                return isolinux, bootcat
        
        for isolinux_file in iso_dir.rglob('isolinux.bin'):
            rel_path = isolinux_file.relative_to(iso_dir)
            return str(rel_path), str(rel_path.parent / 'boot.cat')
        
        return None, None
    
    def build_variant(self, base_iso, lang, arch, name):
        print(f"\n=== {name} ({lang}_{arch}) ===")
        
        with tempfile.TemporaryDirectory() as tmpdir:
            isodir = Path(tmpdir) / 'iso'
            coredir = Path(tmpdir) / 'core_fs'
            
            self.log('info', "iso extracting")
            if not self.extract_iso(base_iso, isodir):
                self.log('err', "Failed to extract ISO")
                return False
            
            self.log('info', "rootfs extracting:")
            core_gz = self.find_core_gz(isodir)
            if not core_gz:
                self.log('err', "no core.gz")
                return False
            
            self.log('info', f"{Path(core_gz).name}")
            if not self.extract_core_gz(core_gz, coredir):
                self.log('warn', "fail")
            
            # 3. Apply overlays
            self.log('info', "overlay applying")
            self.apply_overlays(isodir, coredir, lang, arch)
            
            # 4. Repack core.gz
            self.log('info', "rootfs repacking")
            self.repack_core_gz(coredir, core_gz)
            
            # 5. Find bootloader
            isolinux_bin, boot_cat = self.find_bootloader(isodir)
            self.log('info', f"bootloader: {isolinux_bin or 'not found'}")
            
            # 6. Create ISO
            self.log('info', "final building")
            output_iso = self.output_dir / f"{name}.iso"
            self.create_iso(isodir, output_iso, f"liab-{lang}_{arch}", 
                          isolinux_bin, boot_cat)
            
            self.log('ok', f"done: {output_iso.name}")
        
        return True
    
    def build_all(self):
        print("begin 4 variants build")
        
        configs = [
            (self.iso_32bit, 'e', 'i', 'liab_eng32'),
            (self.iso_32bit, 'r', 'i', 'liab_rus32'), 
            (self.iso_64bit, 'e', 'a', 'liab_eng64'),
            (self.iso_64bit, 'r', 'a', 'liab_rus64')
        ]
        
        success = True
        for base_iso, lang, arch, name in configs:
            if not self.build_variant(base_iso, lang, arch, name):
                success = False
        
        print("\ndone all 4 variants\n===========")
        for iso in self.output_dir.glob('*.iso'):
            print(f"  {iso.name:20} {iso.stat().st_size / 1024 / 1024:.1f} MB")
        
        return success

def main():
    if len(sys.argv) != 3:
        print("Usage: python3 build_tcl.py <32bit.iso> <64bit.iso>")
        sys.exit(1)
    
    iso_32 = sys.argv[1]
    iso_64 = sys.argv[2]
    
    if not os.path.isfile(iso_32):
        print(f"32-bit iso not found: {iso_32}")
        sys.exit(1)
    
    if not os.path.isfile(iso_64):
        print(f"64-bit iso not found: {iso_64}")
        sys.exit(1)
    
    builder = TinyCoreBuilder(iso_32, iso_64)
    builder.build_all()

if __name__ == "__main__":
    main()
