import os
import sys
from pathlib import Path
from typing import List

from flask import Flask

from scanner import ensure_mounted_rw, scan_installs


if getattr(sys, "frozen", False):
    base_dir = sys._MEIPASS  # type: ignore[attr-defined]
else:
    base_dir = os.path.abspath(os.path.dirname(__file__))

static_folder = os.path.join(base_dir, "static")
app = Flask(__name__, static_folder=static_folder)

SYSTEMS: List[dict] = []
MOUNTED_RWS: dict[int, str] = {}

FIRST_REQUEST_SEEN = False
CORE_MODE = "standard" # or "chroot"

def get_core():
  if CORE_MODE == "chroot":
    import linux_chroot_core as core
  else:
    import linux_core as core
  return core


def refresh_systems() -> None:
    global SYSTEMS
    SYSTEMS = scan_installs()


def get_system(idx: int) -> dict:
    if idx < 0 or idx >= len(SYSTEMS):
        raise IndexError
    return SYSTEMS[idx]


def get_system_root_rw(sid: int) -> Path:
    inst = get_system(sid)
    dev = inst["dev"]
    mp = MOUNTED_RWS.get(sid)
    if mp and os.path.isdir(mp):
        return Path(mp)
    mp = ensure_mounted_rw(dev)
    MOUNTED_RWS[sid] = mp
    return Path(mp)


BASE_TEMPLATE = """
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>{{ title }}</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <link rel="stylesheet" href="{{ url_for('static', filename='css/custom.css') }}">
</head>
<body>
<nav class="navbar">
  <div class="navbar-inner">
    <a href="{{ url_for('index') }}" class="navbar-brand">{{ title }}</a>
  </div>
</nav>

<div class="page-container">
  <div class="content-container">
    {{ content|safe }}
  </div>
</div>
</body>
</html>
"""
