import sys
import subprocess
from pathlib import Path
from typing import Dict

#read-only or support
from linux_core import (
read_passwd, read_shadow, read_group, read_file,
detect_admingroup_and_sudoers, password_status, check_user_admin,
write_file, backup_file
)

def run_chroot(root: Path, cmd: list[str], input_data: str = None) -> bool:
    """executes a command inside the chroot environment."""
    full_cmd = ["chroot", str(root)] + cmd
    try:
        res = subprocess.run(
        full_cmd,
        input=input_data,
        capture_output=True,
        text=True,
        timeout=30,
        check=False
        )
        if res.returncode != 0:
            print(f"chroot cmd failed: {' '.join(full_cmd)}\nstdout: {res.stdout}\nstderr: {res.stderr}", file=sys.stderr)
        return res.returncode == 0
    except Exception as e:
        print(f"chroot execution error: {e}", file=sys.stderr)
        return False

def add_user(root: Path, username: str):
    return run_chroot(root, ["useradd", "-M", username])

def set_hashed_password(root: Path, username: str, plain: str):
    return run_chroot(root, ["chpasswd"], input_data=f"{username}:{plain}")

def set_plain_password(root: Path, username: str, plain: str):
    from linux_core import set_plain_password as direct_set
    return direct_set(root, username, plain)

def remove_user(root: Path, username: str):
    return run_chroot(root, ["userdel", "-f", username])

def add_user_to_admingroup(root: Path, admingroup: str, username: str):
    group_db = read_group(root)
    if admingroup not in group_db:
        run_chroot(root, ["groupadd", admingroup])
    return run_chroot(root, ["usermod", "-aG", admingroup, username])
