import sys
import crypt
import shutil
from pathlib import Path
from typing import Dict



def backup_file(path: Path) -> None:
    """/dir/file -> /dir/file.liab.bak"""
    if not path.exists():
        return
    backup = path.with_suffix(path.suffix + ".liab.bak")
    try:
        shutil.copy2(path, backup)
    except Exception as e:
        print(f"backup error {path} -> {backup}: {e}", file=sys.stderr)


def write_lines_unix(path: Path, lines: list[str]) -> None:
    backup_file(path)
    data = ("\n".join(lines) + "\n").encode("utf-8")
    with path.open("wb") as f:  # binmode to fix crlf
        f.write(data)


def write_text_unix(path: Path, content: str) -> None:
    backup_file(path)
    data = content.replace("\r\n", "\n").replace("\r", "\n").encode("utf-8")
    with path.open("wb") as f:
        f.write(data)


def read_passwd(root: Path):
    res: Dict[str, Dict] = {}
    p = root / "etc/passwd"
    if not p.exists():
        return res
    for line in p.read_text(errors="ignore").splitlines():
        if not line or ":" not in line:
            continue
        parts = line.split(":")
        if len(parts) < 7:
            continue
        username, _, uid, gid, gecos, home, shell = parts[:7]
        res[username] = {
            "username": username,
            "uid": int(uid) if uid.isdigit() else -1,
            "gid": int(gid) if gid.isdigit() else -1,
            "gecos": gecos,
            "home": home,
            "shell": shell,
        }
    return res


def read_shadow(root: Path):
    res: Dict[str, Dict] = {}
    p = root / "etc/shadow"
    if not p.exists():
        return res
    for line in p.read_text(errors="ignore").splitlines():
        if not line or ":" not in line:
            continue
        parts = line.split(":")
        if len(parts) < 2:
            continue
        username, pwd = parts[0], parts[1]
        res[username] = {
            "raw": line,
            "password": pwd,
            "parts": parts,
        }
    return res


def read_group(root: Path):
    res: Dict[str, Dict] = {}
    p = root / "etc/group"
    if not p.exists():
        return res
    for line in p.read_text(errors="ignore").splitlines():
        if not line or ":" not in line:
            continue
        parts = line.split(":")
        if len(parts) < 4:
            continue
        name, _, gid, members = parts[:4]
        mem_list = [m for m in members.split(",") if m] if members else []
        res[name] = {
            "name": name,
            "gid": int(gid) if gid.isdigit() else -1,
            "members": mem_list,
            "raw": line,
            "parts": parts,
        }
    return res


def detect_admingroup_and_sudoers(root: Path):
    sudoers = root / "etc/sudoers"
    admingroup = None
    has_entry = False
    if sudoers.exists():
        for line in sudoers.read_text(errors="ignore").splitlines():
            line = line.strip()
            if not line or line.startswith("#"):
                continue
            if line.startswith("%wheel"):
                admingroup = "wheel"
                has_entry = True
            if line.startswith("%sudo"):
                admingroup = "sudo"
                has_entry = True
    return admingroup, has_entry


def check_user_admin(group_db: Dict[str, Dict], admingroup: str, username: str):
    if not admingroup:
        return False
    g = group_db.get(admingroup)
    if not g:
        return False
    return username in g.get("members", [])


def password_status(shadow_db: Dict[str, Dict], username: str):
    rec = shadow_db.get(username)
    if not rec:
        return "none"
    pwd = rec["password"]
    if pwd in ("", None):
        return "empty"
    if pwd in ("*", "!", "!*"):
        return "locked"
    if "$" in pwd:
        return "hashed"
    return pwd


def set_plain_password(root: Path, username: str, plain: str):
    """set password as plain"""
    shadow_path = root / "etc/shadow"
    if not shadow_path.exists():
        return False
    lines = shadow_path.read_text(errors="ignore").splitlines()
    new_lines = []
    modified = False
    for line in lines:
        if not line or ":" not in line:
            new_lines.append(line)
            continue
        parts = line.split(":")
        if parts[0] == username:
            parts[1] = plain
            modified = True
            new_lines.append(":".join(parts))
        else:
            new_lines.append(line)
    if modified:
        write_lines_unix(shadow_path, new_lines)
    return modified


def set_hashed_password(root: Path, username: str, plain: str) -> bool:
    """set password as hash of plain"""
    shadow_path = root / "etc/shadow"
    if not shadow_path.exists():
        return False

    salt = crypt.mksalt(crypt.METHOD_SHA512)
    hashed = crypt.crypt(plain, salt)

    lines = shadow_path.read_text(errors="ignore").splitlines()
    new_lines = []
    modified = False

    for line in lines:
        if not line or ":" not in line:
            new_lines.append(line)
            continue
        parts = line.split(":")
        if parts[0] == username:
            parts[1] = hashed
            modified = True
            new_lines.append(":".join(parts))
        else:
            new_lines.append(line)

    if modified:
        write_lines_unix(shadow_path, new_lines)
    return modified


def remove_user(root: Path, username: str):
    passwd_path = root / "etc/passwd"
    if passwd_path.exists():
        lines = passwd_path.read_text(errors="ignore").splitlines()
        new_lines = [l for l in lines if not l.startswith(username + ":")]
        write_lines_unix(passwd_path, new_lines)

    shadow_path = root / "etc/shadow"
    if shadow_path.exists():
        lines = shadow_path.read_text(errors="ignore").splitlines()
        new_lines = [l for l in lines if not l.startswith(username + ":")]
        write_lines_unix(shadow_path, new_lines)

    group_path = root / "etc/group"
    if group_path.exists():
        lines = group_path.read_text(errors="ignore").splitlines()
        new_lines = []
        for line in lines:
            if not line or ":" not in line:
                new_lines.append(line)
                continue
            parts = line.split(":")
            members = parts[3].split(",") if len(parts) >= 4 and parts[3] else []
            members = [m for m in members if m != username]
            parts[3] = ",".join(members) if members else ""
            new_lines.append(":".join(parts))
        write_lines_unix(group_path, new_lines)

    return True


def add_user_to_admingroup(root: Path, admingroup: str, username: str):
    group_path = root / "etc/group"
    if not group_path.exists():
        return False
    lines = group_path.read_text(errors="ignore").splitlines()
    new_lines = []
    modified = False
    found = False
    for line in lines:
        if not line or ":" not in line:
            new_lines.append(line)
            continue
        parts = line.split(":")
        name = parts[0]
        if name == admingroup:
            found = True
            members = parts[3].split(",") if len(parts) >= 4 and parts[3] else []
            if username not in members:
                members.append(username)
                parts[3] = ",".join(members)
                modified = True
            new_lines.append(":".join(parts))
        else:
            new_lines.append(line)
    if not found:
        new_line = f"{admingroup}:x:27:{username}"
        new_lines.append(new_line)
        modified = True
    if modified:
        write_lines_unix(group_path, new_lines)
    return modified


def read_file(root: Path, rel: str) -> str:
    p = root / rel.lstrip("/")
    if not p.exists():
        return ""
    return p.read_text(errors="ignore")


def write_file(root: Path, rel: str, content: str) -> bool:
    p = root / rel.lstrip("/")
    try:
        write_text_unix(p, content)
        return True
    except Exception as e:
        print(f"write_file error {p}: {e}", file=sys.stderr)
        return False

def add_user(root: Path, username: str):
    passwd_path = root / "etc/passwd"
    shadow_path = root / "etc/shadow"

    if not passwd_path.exists() or not shadow_path.exists():
        return False

    pass_lines = passwd_path.read_text(errors="ignore").splitlines()
    
    uids = []
    for line in pass_lines:
        if line.startswith(f"{username}:"):
            return False
        parts = line.split(":")
        if len(parts) > 2 and parts[2].isdigit():
            uids.append(int(parts[2]))

        next_uid = 1000
    
    if uids:
        filtered_uids = [u for u in uids if 1000 <= u < 60000]
    if filtered_uids:
        next_uid = max(filtered_uids) + 1

    new_passwd = f"{username}:x:{next_uid}:{next_uid}:{username}:/home/{username}:/bin/bash"
    pass_lines.append(new_passwd)
    write_lines_unix(passwd_path, pass_lines)

    shadow_lines = shadow_path.read_text(errors="ignore").splitlines()
    new_shadow = f"{username}:!:18888:0:99999:7:::"
    shadow_lines.append(new_shadow)
    write_lines_unix(shadow_path, shadow_lines)

    group_path = root / "etc/group"
    if group_path.exists():
        group_lines = group_path.read_text(errors="ignore").splitlines()
        new_group = f"{username}:x:{next_uid}:"
        group_lines.append(new_group)
        write_lines_unix(group_path, group_lines)

    return True
