import os
import sys
import json
import subprocess
import tempfile
from pathlib import Path
from typing import Dict, List, Optional

SUPPORTED_FS = {"ext2", "ext3", "ext4", "xfs", "btrfs", "f2fs", "ntfs", "vfat"}


def run_cmd(cmd: List[str], timeout: Optional[int] = None) -> str:
    try:
        res = subprocess.run(cmd, capture_output=True, text=True, timeout=timeout, check=False)
        return res.stdout
    except Exception as e:
        print(f"cmd error {cmd}: {e}", file=sys.stderr)
        return ""


def run_lsblk():
    out = run_cmd(["lsblk", "-J", "-o", "NAME,KNAME,FSTYPE,MOUNTPOINT,TYPE,MOUNTPOINTS,SIZE"])
    if not out:
        return {"blockdevices": []}
    try:
        return json.loads(out)
    except json.JSONDecodeError:
        return {"blockdevices": []}


def collect_partitions():
    data = run_lsblk()
    result: List[Dict] = []

    def walk(nodes):
        for d in nodes:
            if d.get("type") == "part":
                fstype = d.get("fstype")
                if fstype and fstype.lower() in SUPPORTED_FS:
                    result.append(d)
            if "children" in d:
                walk(d["children"])

    walk(data.get("blockdevices", []))
    return result


def get_existing_mountpoint(entry: Dict, dev: str):
    mps = entry.get("mountpoints") or []
    for mp in mps:
        if mp:
            return mp
    mp = entry.get("mountpoint")
    if mp:
        return mp
    out = run_cmd(["findmnt", "-n", "-o", "TARGET", dev])
    for line in out.splitlines():
        line = line.strip()
        if line:
            return line
    out = run_cmd(["mount"])
    for line in out.splitlines():
        if dev in line:
            parts = line.split()
            if len(parts) >= 3:
                return parts[2]
    return None


def ensure_mounted_ro(entry: Dict):
    dev = "/dev/" + entry["kname"]
    mp = get_existing_mountpoint(entry, dev)
    if mp:
        return mp, True
    mount_dir = tempfile.mkdtemp(prefix="rescue-ro-")
    os.makedirs(mount_dir, exist_ok=True)
    subprocess.run(["mount", "-o", "ro", dev, mount_dir], check=False)
    return mount_dir, False


def ensure_mounted_rw(dev: str):
    mount_dir = tempfile.mkdtemp(prefix="rescue-rw-")
    os.makedirs(mount_dir, exist_ok=True)
    subprocess.run(["mount", "-o", "rw", dev, mount_dir], check=False)
    return mount_dir


def umount_path(path: str):
    if not path:
        return
    subprocess.run(["umount", path], check=False)


def parse_os_release(root: Path):
    p = root / "etc/os-release"
    res: Dict[str, str] = {}
    if not p.exists():
        return res
    for line in p.read_text(errors="ignore").splitlines():
        line = line.strip()
        if not line or line.startswith("#") or "=" not in line:
            continue
        k, v = line.split("=", 1)
        v = v.strip().strip('"').strip("'")
        res[k] = v
    return res


def detect_install(mount_point: str):
    root = Path(mount_point)
    if (root / "etc/os-release").exists():
        return "linux"
    if (root / "Windows/System32/config/SAM").exists():
        return "windows"
    return None


def scan_installs():
    installs: List[Dict] = []

    #print("DEBUG collect_partitions:", file=sys.stderr)
    parts = collect_partitions()
       

    for entry in parts:
        #print("  part:", e, file=sys.stderr)
        dev = "/dev/" + entry["kname"]
        mp, already = ensure_mounted_ro(entry)
        kind = detect_install(mp)
        #print("  scanned:", dev, "mp=", mp, "kind=", kind, file=sys.stderr)

        if not kind:
            if not already:
                umount_path(mp)
            continue

        size = entry.get("size") or ""
        info: Dict = {
            "dev": dev,
            "kind": kind,
            "mountpoint": None,
            "os_name": "",
            "os_version": "",
            "size": size,
            "id_like": "",
        }
        root = Path(mp)
        if kind == "linux":
            osinfo = parse_os_release(root)
            info["os_name"] = osinfo.get("NAME", "Linux")
            info["os_version"] = osinfo.get("VERSION", "")
            info["id_like"] = osinfo.get("ID_LIKE", "")
        else:
            info["os_name"] = "Windows"
            info["os_version"] = ""

        installs.append(info)

        if not already:
            umount_path(mp)

    #print("Found", len(installs), "systems", file=sys.stderr)
    return installs