from flask import render_template_string, request, redirect, url_for
from urllib.parse import quote_plus
import subprocess
import tempfile
from pathlib import Path
import threading
import sys


from config import _t
import app
from app import get_core
#from linux_core import (
#    read_passwd, read_shadow, read_group,
#    detect_admingroup_and_sudoers, check_user_admin, password_status,
#    set_plain_password, set_hashed_password, remove_user,
#    add_user_to_admingroup, read_file, write_file, add_user
#)


flask_app = app.app

def run_john_async(unshadow_path: Path, username: str) -> None:
    try:
        jproc = subprocess.Popen(
            [
                "/usr/local/sbin/john",
                f"--wordlist=/usr/share/wordlists/rockyou.txt",
                str(unshadow_path),
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            text=True,
        )
        out, err = jproc.communicate()
        if out:
            print(f"[CRACK] john stdout:\n{out}", file=sys.stderr)
        #if err:
            #print(f"[CRACK] john stderr:\n{err}", file=sys.stderr)
    except Exception as e:
        print(f"[CRACK] worker error for {username}: {e}", file=sys.stderr)


@flask_app.route("/linux/<int:sid>/users")
def linux_users(sid: int):
    core = get_core()
    try:
        inst = app.get_system(sid)
    except IndexError:
        return "System not found", 404
    dev = inst["dev"]
    os_name = inst.get("os_name", "")
    os_version = inst.get("os_version", "")
    size = inst.get("size", "")
    os_line = f"{os_name} {os_version}".strip()
    if size:
        os_line = f"{os_line} {size}"

    root = app.get_system_root_rw(sid)
    passwd_db = core.read_passwd(root)
    shadow_db = core.read_shadow(root)
    group_db = core.read_group(root)
    admingroup, _ = core.detect_admingroup_and_sudoers(root)

    rows = []
    for username, pdata in sorted(passwd_db.items(), key=lambda kv: kv[1]["uid"]):
        uid = pdata["uid"]
        if 1 <= uid <= 999:
            continue
        gid = pdata["gid"]
        ps = core.password_status(shadow_db, username)
        if ps == "empty":
            ps_label = _t['empty']
        elif ps == "locked":
            ps_label = _t['locked']
        elif ps == "hashed":
            ps_label = _t['hashed']
        else:
            ps_label = ps
        is_admin = core.check_user_admin(group_db, admingroup, username)
        admin_label = _t['yes'] if is_admin else _t['no']
        user_url = f"/linux/{sid}/user/{username}"
        rows.append(f"""
        <tr>
          <td>{uid}</td>
          <td>{gid}</td>
          <td><a href="{user_url}" class="link">{username}</a></td>
          <td>{ps_label}</td>
          <td>{admin_label}</td>
        </tr>
        """)

    html = f"""
    <h1 class="page-title">{_t['user_list']} (<code>{dev}</code>)</h1>
    <p class="small-text">{os_line}</p>
    <form method="post" action="/linux/{sid}/adduser" class="row-inline mb-1">
    <div class="col-grow">
        <input type="text" name="username" class="input" placeholder="{_t['new_username']}">
    </div>
    <div class="col-fixed">
        <button class="btn small primary">+</button>
    </div>
    </form>
    <div class="table-wrap">
      <table class="table">
        <thead>
          <tr>
            <th>{_t['uid']}</th>
            <th>{_t['gid']}</th>
            <th>{_t['username']}</th>
            <th>{_t['password_status']}</th>
            <th>{_t['admin_group_member']}</th>
          </tr>
        </thead>
        <tbody>
          {''.join(rows)}
        </tbody>
      </table>
    </div>
    <a href="/system/{sid}" class="btn small mt-1">{_t['back']}</a>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)

@flask_app.route("/linux/<int:sid>/adduser", methods=["POST"])
def linux_adduser(sid: int):
    core = get_core()
    username = request.form.get("username", "").strip()
    if not username:
        msg = _t['operation_failed']
        return redirect(f"/linux/{sid}/users?msg={quote_plus(msg)}")

    root = app.get_system_root_rw(sid)
    ok = core.add_user(root, username)
    msg = _t['operation_ok'] if ok else _t['operation_failed']
    return redirect(f"/linux/{sid}/users?msg={quote_plus(msg)}")


@flask_app.route("/toggle_core")
def toggle_core():
    app.CORE_MODE = "chroot" if app.CORE_MODE == "standard" else "standard"
    return redirect(request.referrer or url_for('index'))

@flask_app.route("/linux/<int:sid>/user/<username>")
def linux_user_page(sid: int, username: str):
    core = get_core()
    try:
        inst = app.get_system(sid)
    except IndexError:
        return "System not found", 404
    dev = inst["dev"]
    root = app.get_system_root_rw(sid)
    admingroup, admingroup_in_sudoers = core.detect_admingroup_and_sudoers(root)
    group_db = core.read_group(root)
    warns = ""
    if admingroup and admingroup not in group_db:
        warns += f"<div class='msg warn-text'>{_t['admingroup_missing']}</div>"
    if admingroup and not admingroup_in_sudoers:
        warns += f"<div class='msg warn-text'>{_t['admingroup_no_sudoers']}</div>"

    msg = request.args.get("msg", "")

    html = f"""
    <h1 class="page-title">{_t['user_actions']}: <code>{username}</code> (<code>{dev}</code>)</h1>
    {warns}
    <div class="msg mb-1">{msg}</div>
    <div class="vstack">
      <form method="post" action="/linux/{sid}/setpwstr" class="card">
        <div class="card-body">
          <input type="hidden" name="username" value="{username}">
          <label>{_t['enter_password']}</label>
          <input type="text" name="pwstr" class="input mb-1">
          <button class="btn small primary">{_t['set_password']}</button>
        </div>
      </form>

      <form method="post" action="/linux/{sid}/setpwhash" class="card">
        <div class="card-body">
          <input type="hidden" name="username" value="{username}">
          <label>{_t['enter_password_twice']}</label>
          <input type="text" name="p1" class="input mb-1">
          <input type="text" name="p2" class="input mb-1">
          <button class="btn small primary">{_t['set_hashed_password']}</button>
        </div>
      </form>

      <form method="post" action="/linux/{sid}/deluser"
            onsubmit="return confirm('Delete user?');"
            class="card">
        <div class="card-body">
          <input type="hidden" name="username" value="{username}">
          <button class="btn small danger">{_t['remove_user']}</button>
        </div>
      </form>

      <form method="post" action="/linux/{sid}/crack" class="card">
        <div class="card-body">
          <input type="hidden" name="username" value="{username}">
            <button class="btn small warn">{_t['crack_password']}</button>
        </div>
      </form>


      <form method="post" action="/linux/{sid}/addadmin" class="card">
        <div class="card-body">
          <input type="hidden" name="username" value="{username}">
          <button class="btn small warn">{_t['add_to_admingroup']}</button>
        </div>
      </form>
    </div>
    <a href="/linux/{sid}/users" class="btn small mt-1">{_t['back']}</a>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)


@flask_app.route("/linux/<int:sid>/setpwstr", methods=["POST"])
def linux_setpass(sid: int):
    core = get_core()
    username = request.form.get("username", "")
    password = request.form.get("pwstr", "")
    root = app.get_system_root_rw(sid)
    ok = core.set_plain_password(root, username, password)
    msg = _t['operation_ok'] if ok else _t['operation_failed']
    return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")


@flask_app.route("/linux/<int:sid>/setpwhash", methods=["POST"])
def linux_sethash(sid: int):
    core = get_core()
    username = request.form.get("username", "")
    p1 = request.form.get("p1", "")
    p2 = request.form.get("p2", "")
    if p1 != p2:
        msg = _t['operation_failed']
        return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")
    root = app.get_system_root_rw(sid)
    ok = core.set_hashed_password(root, username, p1)
    msg = _t['operation_ok'] if ok else _t['operation_failed']
    return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")


@flask_app.route("/linux/<int:sid>/deluser", methods=["POST"])
def linux_deluser(sid: int):
    core = get_core()
    username = request.form.get("username", "")
    root = app.get_system_root_rw(sid)
    ok = core.remove_user(root, username)
    msg = _t['operation_ok'] if ok else _t['operation_failed']
    return redirect(f"/linux/{sid}/users?msg={quote_plus(msg)}")


@flask_app.route("/linux/<int:sid>/addadmin", methods=["POST"])
def linux_addadmin(sid: int):
    core = get_core()
    username = request.form.get("username", "")
    root = app.get_system_root_rw(sid)
    admingroup, _ = core.detect_admingroup_and_sudoers(root)
    if not admingroup:
        admingroup = "sudo"
    ok = core.add_user_to_admingroup(root, admingroup, username)
    msg = _t['operation_ok'] if ok else _t['operation_failed']
    return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")


@flask_app.route("/linux/<int:sid>/files")
def linux_files_menu(sid: int):
    try:
        inst = app.get_system(sid)
    except IndexError:
        return "System not found", 404
    dev = inst["dev"]
    os_name = inst.get("os_name", "")
    os_version = inst.get("os_version", "")
    size = inst.get("size", "")
    os_line = f"{os_name} {os_version}".strip()
    if size:
        os_line = f"{os_line} {size}"

    default_files = [
        "/etc/passwd",
        "/etc/shadow",
        "/etc/group",
        "/etc/sudoers",
        "/etc/os-release",
    ]

    items = []
    for f in default_files:
        items.append(
            f'<li class="list-item">'
            f'  <span>{f}</span>'
            f'  <a class="btn small" href="/linux/{sid}/file/{f.lstrip("/")}">{_t["open_file"]}</a>'
            f'</li>'
        )

    msg = request.args.get("msg", "")

    html = f"""
    <h1 class="page-title">{_t['file_editor']} (<code>{dev}</code>)</h1>
    <p class="small-text">{os_line}</p>
    <div class="msg mb-1">{msg}</div>

    <h2 class="section-title">{_t['standard_files']}</h2>
    <ul class="list">
      {''.join(items)}
    </ul>

    <h2 class="section-title">{_t['arbitrary_file']}</h2>
    <form method="get" action="/linux/{sid}/file_by_path" class="row-inline">
      <div class="col-grow">
        <input type="text" name="path" class="input" placeholder="/etc/hosts">
      </div>
      <div class="col-fixed">
        <button class="btn small primary">{_t['open_file']}</button>
      </div>
    </form>

    <a href="/system/{sid}" class="btn small mt-1">{_t['back']}</a>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)


@flask_app.route("/linux/<int:sid>/file_by_path")
def linux_file_by_path(sid: int):
    path = request.args.get("path", "").strip()
    if not path:
        return redirect(f"/linux/{sid}/files?msg={quote_plus(_t['empty_path'])}")
    rel = path.lstrip("/")
    return redirect(f"/linux/{sid}/file/{rel}")


@flask_app.route("/linux/<int:sid>/file/<path:rel>", methods=["GET", "POST"])
def linux_file_edit(sid: int, rel: str):
    core = get_core()
    inst = app.get_system(sid)
    dev = inst["dev"]
    root = app.get_system_root_rw(sid)
    msg = ""
    if request.method == "POST":
        content = request.form.get("content", "")
        ok = core.write_file(root, rel, content)
        msg = _t['operation_ok'] if ok else _t['operation_failed']
    content = core.read_file(root, rel)
    warn = ""
    if rel.endswith("sudoers"):
        warn = f"<div class='msg warn-text'>{_t['danger_sudoers']}</div>"
    html = f"""
    <h1 class="page-title">/{rel} (<code>{dev}</code>)</h1>
    {warn}
    <div class="msg mb-1">{msg}</div>
    <form method="post">
      <textarea name="content" class="textarea">{content}</textarea>
      <button class="btn small primary mt-1">{_t['submit']}</button>
    </form>
    <a href="/linux/{sid}/files" class="btn small mt-1">{_t['back']}</a>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)

@flask_app.route("/linux/<int:sid>/crack", methods=["POST"])
def linux_crack_password(sid: int):
    username = request.form.get("username", "").strip()
    if not username:
        msg = _t['operation_failed']
        return redirect(f"/linux/{sid}/users?msg={quote_plus(msg)}")

    root = app.get_system_root_rw(sid)
    passwd_path = root / "etc/passwd"
    shadow_path = root / "etc/shadow"

    try:
        with tempfile.NamedTemporaryFile(prefix="liab_unshadow_", suffix=".txt", delete=False) as tf:
            tmp_unshadow = Path(tf.name)

        proc = subprocess.run(
            ["/usr/local/sbin/unshadow", str(passwd_path), str(shadow_path)],
            check=False,
            capture_output=True,
            text=True,
        )
        if proc.returncode != 0:
            print(f"unshadow failed: {proc.returncode}, {proc.stderr}", file=sys.stderr)
            msg = _t['operation_failed']
            return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")

        lines = [ln for ln in proc.stdout.splitlines() if ln.startswith(username + ":")]
        if not lines:
            msg = _t['operation_failed']
            return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")

        tmp_unshadow.write_text("\n".join(lines) + "\n")

    except Exception as e:
        print(f"linux_crack_password: error preparing unshadow file: {e}", file=sys.stderr)
        msg = _t['operation_failed']
        return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")

    try:
        t = threading.Thread(
            target=run_john_async,
            args=(tmp_unshadow, username),
            daemon=True,
        )
        t.start()
        msg = _t['crack_started']
    except Exception as e:
        print(f"linux_crack_password: error starting worker: {e}", file=sys.stderr)
        msg = _t['operation_failed']

    return redirect(f"/linux/{sid}/user/{username}?msg={quote_plus(msg)}")
