import sys
import subprocess
from pathlib import Path

from flask import render_template_string, request
from config import _t

import app
flask_app = app.app


@flask_app.route("/")
def index():
    app.refresh_systems()
    if not app.SYSTEMS:
        html = f"""
        <div class="center-text">
          <p class="lead">{_t['no_systems']}</p>
        </div>
        """
        return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)

    items = []
    for i, inst in enumerate(app.SYSTEMS):
        dev = inst["dev"]
        kind = inst["kind"]
        os_name = inst.get("os_name", "")
        os_version = inst.get("os_version", "")
        size = inst.get("size", "")
        kind_label = "GNU/Linux" if kind == "linux" else _t['windows']

        os_line = f"{os_name} {os_version}".strip()
        if size:
            os_line = f"{os_line} {size}"

        url = f"/system/{i}"
        items.append(
            f'<li class="list-item">'
            f'  <div class="item-main">'
            f'    <span><code>{dev}</code> {kind_label}</span>'
            f'    <small>{os_line}</small>'
            f'  </div>'
            f'  <a href="{url}" class="btn small">{_t["open"]}</a>'
            f'</li>'
        )

    html = f"""
    <div>
    <span>{_t['core']}: <a href="/toggle_core" class="btn small"><code>{app.CORE_MODE}</code></a></span>
    </div>
    <h2 class="title-center">{_t['system_list']}</h2>
    <ul class="list">
      {''.join(items)}
    </ul>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)


@flask_app.route("/system/<int:sid>", methods=["GET", "POST"])
def system_page(sid: int):
    try:
        inst = app.get_system(sid)
    except IndexError:
        return "System not found", 404

    dev = inst["dev"]
    kind = inst["kind"]
    os_name = inst.get("os_name", "")
    os_version = inst.get("os_version", "")
    size = inst.get("size", "")
    id_like = inst.get("id_like", "")

    root = app.get_system_root_rw(sid)
    mp = str(root)

    os_line = f"{os_name} {os_version}".strip()
    if size:
        os_line = f"{os_line} {size}"
    id_like_line = f"ID_LIKE: {id_like}" if (id_like and kind == "linux") else ""

    cmd_output = ""
    if request.method == "POST" and kind == "linux":
        cmd_str = request.form.get("cmd", "")
        if cmd_str:
            try:
                cmd = ["chroot", mp] + cmd_str.strip().split()
                res = subprocess.run(
                    cmd,
                    capture_output=True,
                    text=True,
                    timeout=30,
                    check=False,
                )
                cmd_output = f"$ {' '.join(cmd)}\n\nSTDOUT:\n{res.stdout}\n\nSTDERR:\n{res.stderr}"
            except Exception as e:
                cmd_output = f"$ {cmd_str}\n\nERROR: {e}"

    from linux_core import detect_admingroup_and_sudoers

    if kind == "windows":
        sam = str(root / "Windows/System32/config/SAM")
        html = f"""
        <h1 class="page-title">{_t['windows']}: <code>{dev}</code></h1>
        <p class="small-text">{os_line}</p>
        <p>{_t['unsupported_windows']}</p>
        <p>SAM: <code>{sam}</code></p>
        <form method="post" action="/run_chntpw/{sid}">
          <button class="btn warn">{_t['run_chntpw']}</button>
        </form>
        <a href="/" class="btn small">{_t['back']}</a>
        """
        return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)

    admingroup, admingroup_in_sudoers = detect_admingroup_and_sudoers(root)
    warns = ""
    group_text = (root / "etc/group").read_text(errors="ignore") if (root / "etc/group").exists() else ""
    if admingroup and admingroup + ":" not in group_text:
        warns += f"<div class='msg warn-text'>{_t['admingroup_missing']}</div>"
    if admingroup and not admingroup_in_sudoers:
        warns += f"<div class='msg warn-text'>{_t['admingroup_no_sudoers']}</div>"

    cmd_block = f"""
    <div class="card">
      <div class="card-header">
        <span class="card-title">{_t['command']}</span>
      </div>
      <div class="card-body">
        <form method="post" class="row-inline">
          <div class="col-grow">
            <input type="text" name="cmd" class="input"
                   placeholder="ls -l /home">
          </div>
          <div class="col-fixed">
            <button class="btn small primary">{_t['run']}</button>
          </div>
        </form>
    """
    if cmd_output:
        cmd_block += f"<pre class='pre-output'>{cmd_output}</pre>"
    cmd_block += "</div></div>"

    html = f"""
    <h1 class="page-title">{_t['linux']}: <code>{dev}</code></h1>
    <p class="small-text">{os_line}</p>
    {f'<p class="small-text">{id_like_line}</p>' if id_like_line else ''}
    {warns}
    <div class="row-inline mb-1">
      <a class="btn small" href="/linux/{sid}/users">{_t['users']}</a>
      <a class="btn small" href="/linux/{sid}/files">{_t['file_editor']}</a>
    </div>
    {cmd_block}
    <div class="mt-1">
      <a href="/" class="btn small">{_t['back']}</a>
    </div>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)


@flask_app.route("/run_chntpw/<int:sid>", methods=["POST"])
def run_chntpw(sid: int):
    try:
        inst = app.get_system(sid)
    except IndexError:
        return "System not found", 404
    root = app.get_system_root_rw(sid)
    sam = str(root / "Windows/System32/config/SAM")
    try:
        subprocess.Popen(["chntpw", "-i", sam])
        msg = _t['operation_ok']
    except Exception as e:
        msg = f"{_t['operation_failed']}: {e}"
    html = f"""
    <h1 class="page-title">chntpw</h1>
    <div class="msg">{msg}</div>
    <a href="/" class="btn small">{_t['back']}</a>
    """
    return render_template_string(app.BASE_TEMPLATE, title=_t['title'], content=html)
